' LoadTGABank:TBank(filename:String,offset:Int=0)
' LoadPixmapBank:TBank(filename:String)
' SaveImageBank:TBank(bank:TBank,filename:String)
' GetBankPixel:Int(bank:TBank,px:Int,py:Int,width:Int,offset:Int=0)
' SetBankPixel(bank:TBank,value:Int,px:Int,py:Int,width:Int)
' CreateImageFromBank:TPixmap(bank:TBank,width:Int,height:Int,alphaflag:Int=False)
' ScaleBank:TBank(bank:TBank,width:Int,height:Int,ratio:Float)
' Bilinear:Int(bank:TBank,width:Int,height:Int,x:Float,y:Float)


Global LOD:Int = 1024

' ----------------------------------------------------------------------------
' Load a greyscale TGA Imagebank (8bit, top left, uncompressed)
' ----------------------------------------------------------------------------
Function LoadTGABank:TBank(filename:String, offset:Int = 0)

	Local f:TStream = OpenStream(filename)
	Local bank:TBank = CreateBank(LOD * LOD / 2)

	' seek to dimension offset
	f.Seek(12)

	' read image dimensions
	Local width:Int = ReadShort(f)
	Local Height:Int = ReadShort(f)
	
	If f Then

		' seek to data offset
		f.Seek(18)

		' copy stream to bank
		ReadBank(bank, f, 0, width * Height)

	EndIf

	Return bank
	
	CloseFile f

End Function



' ----------------------------------------------------------------------------
' Load a normal Imagebank
' ----------------------------------------------------------------------------
Function LoadPixmapBank:TBank(filename:String)

	Local pixmap:TPixmap = LoadPixmap(filename)
	Local width:Int = pixmap.width
	Local Height:Int = pixmap.Height
	Local bank:TBank = CreateBank(width * Height)

	For Local y:Int = 0 To Height - 1

		For Local x:Int = 0 To width - 1

			SetBankPixel(bank, ReadPixel(pixmap, x, y) Shr 16 & %11111111, x, y, width)

	Next

	Next

	pixmap = Null

	Return bank
	
End Function



' ----------------------------------------------------------------------------
' Save a RAW Imagebank
' ----------------------------------------------------------------------------
Function SaveImageBank:TBank(bank:TBank, filename:String)
	
	Local f:TStream = WriteFile(filename)
	
	WriteBank(bank, f, 0, BankSize(bank))
	
	CloseFile f
	
End Function



' ----------------------------------------------------------------------------
' Reads a RAW Imagebank Pixel at a given position
' ----------------------------------------------------------------------------
Function GetBankPixel:Int(bank:TBank, px:Int, py:Int, width:Int, offset:Int = 0)
	
	Return PeekByte(bank, ((py * width) + px + offset) Mod BankSize(bank))
	
End Function

' ----------------------------------------------------------------------------
' Writes a RAW Imagebank Pixel at a given position
' ----------------------------------------------------------------------------
Function SetBankPixel(bank:TBank, Value:Int, px:Int, py:Int, width:Int)
	
	PokeByte(bank, (py * width) + px, Value Mod BankSize(bank))
	
End Function

' ----------------------------------------------------------------------------
' Create a greyscale Image from a RAW Imagebank
' ----------------------------------------------------------------------------
Function CreateImageFromBank:TPixmap(bank:TBank, width:Int, Height:Int, alphaflag:Int = False)
	
	Local x:Int, y:Int, h:Int, Alpha:Int
	Local pixmap:TPixmap
	
	If alphaflag = True Then

		pixmap = CreatePixmap(width, Height, PF_A8)

	Else

		pixmap = CreatePixmap(width, Height, PF_I8)

	EndIf
	
	For x = 0 To width - 1
		
		For y = 0 To Height - 1
			
			h = GetBankPixel(bank, x, y, width)
			
			If alphaflag Then Alpha = alphaflag Else Alpha = h
			WritePixel(pixmap, x, y, h | (h Shl 8) | (h Shl 16) | (Alpha Shl 24))
			
		Next
		
	Next
	
	Return pixmap
	
End Function

' ----------------------------------------------------------------------------
' Bilinear scale a RAW Imagebank To a different size using a ratio variable
' ----------------------------------------------------------------------------
Function ScaleBank:TBank(bank:TBank, width:Int, Height:Int, ratio:Float)
	
	Local x:Int, y:Int, c:Int
	Local ratio2w:Float = (ratio:Float * width - 1) / (width - 1.0)
	Local ratio2h:Float = (ratio:Float * Height - 1) / (Height - 1.0)
	
	Local newbank:TBank = CreateBank((width * ratio) * (Height * ratio))
	
	For y = 0 To (Height * ratio) - 1
		
		For x = 0 To (width * ratio) - 1
			
			Local u:Float = x * 1.0 / ratio2w:Float
			Local v:Float = y * 1.0 / ratio2h:Float
			
			c:Int = Bilinear(bank, width, Height, u, v)
			
			SetBankPixel(newbank, c, x, y, width * ratio)
			
		Next
		
	Next
	
	Return newbank
	
End Function

' ----------------------------------------------------------------------------
' Bilinear Scale Function
' ----------------------------------------------------------------------------
Function Bilinear:Int(bank:TBank, width:Int, Height:Int, x:Float, y:Float)
	
	Local x0:Float = Floor(x)
	Local y0:Float = Floor(y)
	Local x1:Float = Floor(x + 1)
	Local y1:Float = Floor(y + 1)
	Local dx:Float = x - x0
	Local dy:Float = y - y0
	
	Local x02x:Int = 1
	Local x03x:Int = 1
	Local y02x:Int = 1
	Local y03x:Int = 1
	
	If x0 < 0 Then x02x = 0
	If x1 > width - 1 Then x03x = 0
	If y0 < 0 Then y02x = 0
	If y1 > Height - 1 Then y03x = 0
	
	If x02x = 0 Then x0 = 0
	If y02x = 0 Then y0 = 0
	If x03x = 0 Then x1 = 0
	If y03x = 0 Then y1 = 0
	
	Local r0:Int = GetBankPixel(bank, x0, y0, width) * x02x * y02x
	Local r1:Int = GetBankPixel(bank, x1, y0, width) * x03x * y02x
	Local r2:Int = GetBankPixel(bank, x1, y1, width) * x03x * y03x
	Local r3:Int = GetBankPixel(bank, x0, y1, width) * x02x * y03x
	
	Return LinearInterpolate(LinearInterpolate(r0, r1, dx), LinearInterpolate(r3, r2, dx), dy)
		
End Function