SuperStrict

Import "gfx/blitz.o"

Framework brl.GLMax2D

Import sidesign.MiniB3D

Import brl.tgaloader
Import brl.pngloader
Import brl.oggloader
Import brl.wavloader
Import brl.jpgloader

Import brl.Pixmap
Import brl.Random
Import brl.timer
Import brl.EventQueue
Import brl.DirectSoundAudio
Import brl.FreeTypeFont
Import brl.Map

Import bah.DBSQLite													' SQlite database extension
Import bah.RegEx													' Regular Expression extension

Import maxgui.Drivers
Import maxgui.ProxyGadgets
Import maxgui.localization

Const SHOWNAMES:Int = 1
Const SHOWLINES:Int = 0

Const SHOWNEBULA:Int = 1
Const SHOWDUST:Int = 1
Const SHOWSTARS:Int = 1

Const STARS:Int = 10000
Const NEBULAE:Int = 1000
Const DUST:Int = 500

Global selectedquery:Int = 5
Global elitediff:Float = 284
Global elitescaler:Float = 88.0
Global uvx:Float = -79
Global uvy:Float = -21
Global uvs:Float = 7031

Global initialgalaxytex:String = "textures/galaxy_1k.jpg"
Global initialcmdr:String = "ESA.db"

Global CAMERAMODE:Int = 0
Global CameraModes:String[] = ["3D Freeflight", "2D Topdown"]

' ------------------------------------------------------------------------------------------------
' Audio Driver
' ------------------------------------------------------------------------------------------------
?Win32
SetAudioDriver "directsound"
?MacOs
SetAudioDriver "freeaudio"
?

' ------------------------------------------------------------------------------------------------
' Sourcecode Include
' ------------------------------------------------------------------------------------------------

Include "include\System\GUI.bmx"
Include "include\System\Init.bmx"
Include "include\System\Media.bmx"
Include "include\System\Objects.bmx"
Include "include\System\System.bmx"
Include "include\System\Variables.bmx"

Include "include\Functions\F3D.bmx"
Include "include\Functions\FBanks.bmx"
Include "include\Functions\FLog.bmx"
Include "include\Functions\FMath.bmx"
Include "include\Functions\FMouse.bmx"
Include "include\Functions\FNoise.bmx"
Include "include\Functions\FPixel.bmx"
Include "include\Functions\FPixmap.bmx"
Include "include\Functions\FQuaternion.bmx"
Include "include\Functions\FRandom.bmx"
Include "include\Functions\FText.bmx"
Include "include\Functions\FWin32.bmx"
Include "include\Functions\FWorld.bmx"

Include "include\Types\TBeat.bmx"
Include "include\Types\TCam.bmx"
Include "include\Types\TConfig.bmx"
Include "include\Types\TFlag.bmx"
Include "include\Types\TFPS.bmx"
Include "include\Types\TLocale.bmx"
Include "include\Types\TMedia.bmx"
Include "include\Types\TQuad.bmx"
Include "include\Types\TSimplex.bmx"
Include "include\Types\TScenegraph.bmx"
Include "include\Types\TTime.bmx"

Type TPath

	Field id:Int
	Field x:Float
	Field y:Float
	Field z:Float
	Field q:TQuad
	
	Field starname:String

End Type

Type TESAQuery

	Field id:Int
	Field title:String
	Field desc:String
	Field sql:String

End Type

' --------------------------------------------------------------------------------
' Variables
' --------------------------------------------------------------------------------

' Database
Global db:TDBConnection
Global q:TDatabaseQuery

' Paths, Lists and Maps
Global p:TPath
Global galaxy:TMesh
Global bubble:TMesh
Global center:TMesh
Global raster1:TMesh
Global raster2:TMesh
Global raster3:TMesh
Global galaxytex:TTexture

Global querymap:TMap = CreateMap()

Global starlist:TList = CreateList()
Global nebulalist:TList = CreateList()
Global dustlist:TList = CreateList()

Global list:TList = CreateList()
Global tourlist:TList = CreateList()
Global vislist:TList = CreateList()
Global systemlist:TList = CreateList()
Global connectlist:TList = CreateList()

' Counter
Global mx:Int
Global start:Int
Global cn:Int

Global oldx:Float
Global oldy:Float
Global oldz:Float

' 3D
Global qx:TQuad
Global vx:TQuad
Global qc:TQuad
Global player:TMesh
Global dummy:TMesh
Global xmesh:TMesh
Global xsurf:TSurface
Global cmesh:TMesh
Global csurf:TSurface
Global qmesh:TMesh
Global qsurf:TSurface
Global nmesh:TMesh
Global nsurf:TSurface
Global dmesh:TMesh
Global dsurf:TSurface

' Timing
Global TICK:TTimer = CreateTimer(1000)
Global GUITIMER:TTimer = CreateTimer(1)
Global TARGETTIMER:TTimer = CreateTimer(TARGET_FPS)

' Flags
Global MOUSE:Int
Global MOUSECLICKED:Int
Global DIRECTION:Int = 0
Global MOUSEMODE:Int = 0

' star cubes dimension
Global qmeshes:TEntity[99, 99, 99]
Global stp:Int = 8

' MAXGUI
Global textfield:TGadget
Global gadgetfont:TGuiFont
Global pulldown:TGadget
Global buttonquery:TGadget
Global buttonloadcmdr:TGadget
Global buttonloadtex:TGadget
Global desc:String



' ------------------------------------------------------------------------------------------------
' Initialize
' ------------------------------------------------------------------------------------------------
AppTitle = PROJECT_TITLE
GLShareContexts()
SeedRnd 42

' Load Database file

' 3D and GUI setup
InitGame()
InitScene()
InitGalaxy()
InitGUI()

' create the 3D Tour
InitTour(initialcmdr)
ResetTour(TESAQuery(MapValueForKey(querymap, String(selectedquery))).sql)


' ------------------------------------------------------------------------------------------------
' Main Loop
' ------------------------------------------------------------------------------------------------
Global initializing:Int = MilliSecs()
AddHook EmitEventHook, EventHook

While Not AppTerminate()

Wend

End



' ----------------------------------------------------------------------------
' create a new document within a tab
' ----------------------------------------------------------------------------
Function CreateDocument:TGadget(tabber:TGadget)

	Local panel:TGadget
	panel = CreatePanel(0, 0, ClientWidth(tabber), ClientHeight(tabber), tabber)
	SetGadgetLayout panel,1,1,1,1
	HideGadget panel
	
	Return panel
	
End Function



' ------------------------------------------------------------------------------------------------
' Update Game Logic
' ------------------------------------------------------------------------------------------------
Function MainLoop()
	
	' read current time
	TIMER_Timer = MilliSecs()

	' check if update is allowed
	If (TIMER_Timer > TIMER_Update + TIMER_Lasttimer) Then

		' calculate new timer interval		
		TIMER_Interval = TIMER_Timer - TIMER_Lasttimer

		If TIMER_Interval > TIMER_Safe Then

			' check for framedrops
			TIMER_Interval = TIMER_Safe
			FPS.framedrops:+1

		End If
		
		FPS.multi = GAME_SPEED * TIMER_Interval
		
		' store last timer
		TIMER_Lasttimer = TIMER_Timer
			
	EndIf

End Function



' ------------------------------------------------------------------------------------------------
' Events
' ------------------------------------------------------------------------------------------------
Function EventHook:Object(id:Int, Data:Object, Context:Object)

	Local Event:TEvent = TEvent(data)
	If Event = Null Then Return Event

	Select Event.id

		Case EVENT_WINDOWCLOSE
								End

		Case EVENT_APPTERMINATE
								End

		Case EVENT_TIMERTICK
								' Update GUI elements

								
								Select Event.Source
								
									Case SCREEN_TIMER
														
														' Update Camera
														If MOUSEMODE = 1 Then
														
															CAMERA.Movement
															
															
														EndIf
														
														CAMERA.zoom = 1.0
														CAMERA.Update()
														CAMERA.GetCameraPosition
															
														Select DIRECTION
														
															Case 1
																		If TICK.Ticks() Mod TARGET_FPS = 0 Then cn:+MouseWheel
																		If cn > mx Then cn = mx
																		
															Case 2
																		If TICK.Ticks() Mod TARGET_FPS = 0 Then cn:-MouseWheel
																		If cn < 1 Then cn = 0
																		
														End Select
														
									Case CANVAS_TIMER
														
														RedrawGadget MAXGUI_CANVAS
														
									Case GUITIMER
														RedrawGadget pulldown
														RedrawGadget textfield
														RedrawGadget buttonquery
														RedrawGadget buttonloadcmdr
														RedrawGadget buttonloadtex

								End Select

		Case EVENT_GADGETACTION
								
								Select Event.source
								
									Case pulldown
								
													Local sel:Int = SelectedGadgetItem(pulldown) + 1
													Local e:TESAQuery = TESAQuery(MapValueForKey(querymap, String(sel)))
													SetGadgetText(textfield, e.sql)
													desc = e.desc
													RedrawGadget(textfield)
													ResetTour(e.sql)
													
									Case buttonquery
									
													ResetTour(GadgetText(textfield))
													
									Case buttonloadcmdr
													
													Local cmdr:String = RequestFile("Select CMDR SQlite database to visualize", "SQlite DB:db;All Files:*", False, "")
													InitTour(cmdr)
													ResetTour(TESAQuery(MapValueForKey(querymap, String(selectedquery))).sql)
													
									Case buttonloadtex
													
													Local tex:String = RequestFile("Select Texture file to load", "Textures:jpg,png,tga;All Files:*", False, "textures/")
													
													FreeEntity galaxy
													FreeTexture galaxytex
													CreateGalaxy(tex)
													
								End Select

		Case EVENT_GADGETPAINT
								
								SetGraphics CanvasGraphics(MAXGUI_CANVAS)
								
								MainLoop()
								
								If SHOWSTARS Then UpdateStars()
								If SHOWNEBULA Then UpdateNebulae()
								If SHOWDUST Then UpdateDust()
								
								UpdateStarBoxes()
								UpdateGalaxy()
								UpdateTour()
					
								FPS.Update()
								
								If CAMERAMODE = 1 Then CAMERA.pitch = 90 ; CAMERA.yaw = 0 ; CAMERA.roll = 0
								
								RenderWorld

								If SHOWLINES Then DrawSystemLines()

								BeginMax2D()

									DrawText desc, 205, 25
									
									DrawText "MOD: " + CameraModes[CAMERAMODE], 0, 25
									DrawText "FPS: " + FPS.FPS, 0, 40
									DrawText "POS: " + cn, 0, 55
									DrawText "INC: " + MouseWheel, 0, 70
									
									If SHOWNAMES Then DrawSystemNames()
	
									Local i1:Int = cn - 1
									If i1 < 0 Then i1 = 0

									p = TPath(list.ValueAtIndex(i1))
									PositionEntity player, p.x / elitescaler, p.y / elitescaler, p.z / elitescaler

									HighlightPOI(p.starname, p.x, p.y, p.z)
									HighlightPOI("Sol", 0, 0, 0)
									HighlightPOI("Colonia", -9530.5, -910.25125, 19808.125)
									HighlightPOI("Beagle Point", -1111.5625, -134.21875, 65269.75)
									HighlightPOI("Praei Anomaly", -1000.375, 12.5625, 54000.46875)
									HighlightPOI("Sagittarius A*", 25.21875, -20.90625, 25899.96875)
									HighlightPOI("Eta Carinae", 7339.25, -84.625, 2327.6875)
									HighlightPOI("VV Cephei", -2348.34375, 304.71875, -637.6875)

								EndMax2D()

								' Bufferflip
								Flip FLAG.Get("VSYNC")

								' Memory Garbage Collection
								GCCollect()

								' let the OS breathe
								Delay FLAG.Get("DELAY")


		Case EVENT_KEYDOWN
								
								Select Event.Data

									' Player Movement
									Case KEY_W
													If CAMERAMODE = 0 Then PlayerZSpeed = 1 Else PLAYERYSpeed = 1
									Case KEY_S
													If CAMERAMODE = 0 Then PlayerZSpeed = -1 Else PLAYERYSpeed = -1
									Case KEY_A
													PlayerVXSpeed = -1
									Case KEY_D
													PlayerVXSpeed = 1
									Case KEY_R
													If CAMERAMODE = 0 Then PlayerYSpeed = 1 Else PLAYERZSpeed = 1
									Case KEY_F
													If CAMERAMODE = 0 Then PlayerYSpeed = -1 Else PLAYERZSpeed = -1
									Case KEY_Q
													PlayerVXSpeed = -1
									Case KEY_E
													PlayerVXSpeed = 1
									Case KEY_X
													PlayerBrake = 1
													PlayerZSpeed = 0
													PlayerYSpeed = 0
													PlayerXSpeed = 0
													
									Case KEY_LSHIFT
													PlayerTurbo = 5
													
									Case KEY_SPACE
													start = 1
													
									Case KEY_UP
													DIRECTION = 1

									Case KEY_DOWN
													DIRECTION = 2
													
									' System
									Case KEY_ESCAPE
													End
													
									Case KEY_TAB
													FLAG.Switch("WIREFRAME")
													
													
								End Select

		Case EVENT_KEYUP
								DIRECTION = 0
								
								Select Event.Data
								
									' Player Movement
									Case KEY_W
													If CAMERAMODE = 0 Then PlayerZSpeed = 0 Else PLAYERYSpeed = 0
													
									Case KEY_S
													If CAMERAMODE = 0 Then PlayerZSpeed = 0 Else PLAYERYSpeed = 0
													
									Case KEY_A
													PlayerVXSpeed = 0
													
									Case KEY_D
													PlayerVXSpeed = 0

									Case KEY_R
													If CAMERAMODE = 0 Then PlayerYSpeed = 0 Else PLAYERZSpeed = 0
													
									Case KEY_F
													If CAMERAMODE = 0 Then PlayerYSpeed = 0 Else PLAYERZSpeed = 0
													
									Case KEY_Q
													PlayerVXSpeed = 0
													
									Case KEY_E
													PlayerVXSpeed = 0
													
									Case KEY_X
													PlayerBrake = 0
													
									Case KEY_LSHIFT
													PlayerTurbo = 0
													
								End Select
								
		Case EVENT_MOUSEWHEEL
						
								If Event.Data > 0 Then

									MouseWheel:+1
									If MouseWheel > 50 Then MouseWheel = 50

								Else

									MouseWheel:-1
									If MouseWheel < 1 Then MouseWheel = 1

								EndIf


		Case EVENT_MOUSEMOVE

								If MOUSEMODE = 1 Then
								
									MOUSECLICKED = 0
									CAMERA.mx = Event.x
									CAMERA.MY = Event.y

								Else
								
									Camera.Stop()
									
								EndIf
								
		Case EVENT_MOUSEDOWN
		
								MOUSEMODE = 1
								
								If MOUSECLICKED = 0 Then
								
									MoveMouse(SCREEN_WIDTH / 2, SCREEN_HEIGHT / 2)
									
									'Camera.center()
									MOUSECLICKED = 1
									
								EndIf
								
								If Event.Data = 2 Then
								
									CAMERAMODE = 1 - CAMERAMODE
									
									If CAMERAMODE = 1 Then Camera.pitch = 45
									
								EndIf
								
								
		Case EVENT_MOUSEUP
		
								MOUSEMODE = 0
								MOUSECLICKED = 0
								CAMERA.Stop()

	End Select
	
End Function

Function UpdateTour()

	For Local x:TQuad = EachIn tourlist

		If MilliSecs() > x.updatetime Then
		
			x.distance = Distance3D(x.x, x.y, x.z, CAMERA.x, CAMERA.y, CAMERA.z)
			x.d1 = x.distance
			x.d2 = Distance3D(oldx, oldy, oldz, CAMERA.x, CAMERA.y, CAMERA.z)	
			x.updatetime = MilliSecs() + Rand(0, Sqr(x.id))
			
		EndIf
		
		x.scalex = x.distance / Sqr(x.distance * 5000)
		x.scaley = x.scalex
		x.Update(CAMERA.cam)
		
	Next

End Function

Function UpdateGalaxy()

	' Galaxy Texture Alpha
	Local a:Float = Normalize(Distance3D(0, 0, 0, CAMERA.x, CAMERA.y, CAMERA.z), 0, 512, 0.1, 0.25)
	If a > 0.25 Then a = 0.25
	If a < 0.1 Then a = 0.1
	If TEntity(galaxy) Then EntityAlpha galaxy, 1.0 - (1.0 / (1.0 + Abs(CAMERA.y / 10.0)))

End Function

Function UpdateStarBoxes()

	' distance alpha of star boxes
	For Local x:Int = -stp To stp Step 1
		
		For Local z:Int = -stp To stp Step 1
											
			Local h1:Float = Normalize(Distance3D(camera.X, camera.Y, camera.z, EntityX(qmeshes[x + stp, 0, z + stp]), EntityY(qmeshes[x + stp, 0, z + stp]), EntityZ(qmeshes[x + stp, 0, z + stp])), 0, 1000, 1, 0)
			
			HideEntity qmeshes[x + stp, 0, z + stp]
			
			If h1 < 0 Then h1 = 0
			
			If h1 > 0.8 Then
			
				ShowEntity qmeshes[x + stp, 0, z + stp]
				EntityAlpha qmeshes[x + stp, 0, z + stp], h1
				
			EndIf
	
		Next
		
	Next

End Function

Function UpdateStars()

	For Local s:TQuad = EachIn starlist
	
		If MilliSecs() > s.updatetime Then s.updatetime = MilliSecs() + (Rand(0, s.id) * 100)
		s.Update(CAMERA.cam)

	Next

End Function

Function UpdateNebulae()

	For Local n:TQuad = EachIn nebulalist

		If MilliSecs() > n.updatetime Then n.updatetime = MilliSecs() + (Rand(0, n.id) * 100)
		n.UpdateRotation(CAMERA.cam)

	Next

End Function

Function UpdateDust()

	For Local d:TQuad = EachIn dustlist

		If MilliSecs() > d.updatetime Then d.updatetime = MilliSecs() + (Rand(0, d.id) * 100)
		d.Update(CAMERA.cam)
		
	Next

End Function

' ------------------------------------------------------------------------------------------------
' Draws the System Line connections, but only if you are close to them
' ------------------------------------------------------------------------------------------------
Function DrawSystemLines()

	' prepare OpenGL to draw lines
	glEnable(GL_BLEND)
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA)
	glEnable(GL_COLOR_MATERIAL)
	glLineWidth(1.0)
	glDepthMask(GL_FALSE)
	glBegin(GL_LINES)
	
	Local lc:Int = 0

	For Local x:TQuad = EachIn tourlist
		
		If x.d1 < 100 And x.d2 < 100 Then
		
			TFormPoint(x.x, x.y, x.z, Null, CAMERA.cam)
			
			If TFormedZ() > 0 Then
	
				If lc = 0 Then oldx = x.x ; oldy = x.y ; oldz = x.z
	
				Local a:Float = 1.0 / x.d1
				SetAlpha(a * a)
				Line3D(x.x, x.y, x.z, oldx, oldy, oldz, 255, 255, 255, 2.0 / x.d1)
				
				oldx = x.x
				oldy = x.y
				oldz = x.z

			EndIf

		EndIf

		lc:+1

	Next

	' reset OpenGL
	glDisable(GL_COLOR_MATERIAL)
	glDepthMask(GL_TRUE)
	glEnd()

End Function


' ------------------------------------------------------------------------------------------------
' Draws the highlighted System names, but only if you are close to them
' ------------------------------------------------------------------------------------------------
Function DrawSystemNames()

	SetColor(255, 255, 255)

	For Local x:TQuad = EachIn vislist
	
		If MilliSecs() > x.updatetime Then
		
			x.distance = Distance3D(x.x, x.y, x.z, CAMERA.x, CAMERA.y, CAMERA.z)
			x.updatetime = MilliSecs() + Rand(0, Sqr(x.id))
			
		EndIf
		
		If x.distance < 5 Then
		
			TFormPoint(x.x, x.y, x.z, Null, CAMERA.cam)
			
			If TFormedZ() > 0 Then
	
				CameraProject(CAMERA.cam, x.x, x.y, x.z)
				Local px1:Int = ProjectedX()
				Local py1:Int = ProjectedY()
				
				If (ProjectedZ() >= 0 And px1 > 0 And px1 < SCREEN_WIDTH And py1 > 0 And py1 < SCREEN_HEIGHT) Then
				
					Local a:Float = 1.0 / x.distance
					SetAlpha(a * a)
					DrawText(x.cat, px1 - TextWidth(x.cat) / 2, py1 + TextHeight(x.cat))
					
				EndIf
				
			EndIf
			
		EndIf
		
	Next
	
	SetAlpha(1.0)

End Function



' ------------------------------------------------------------------------------------------------
' Highlights a single POI position XYZ
' ------------------------------------------------------------------------------------------------
Function HighlightPOI(systemname:String, px:Float, py:Float, pz:Float)

	Local x:Int
	Local y:Int

	px:/elitescaler
	py:/elitescaler
	pz:/elitescaler
	
	Local d:Float = Distance3d(px, py, pz, camera.x, camera.y, camera.z)
	TFormPoint(px, py, pz, Null, CAMERA.cam)
	If TFormedZ() > 0 And d < 1000 Then
	
		' current Position
		CameraProject(CAMERA.cam, px, py, pz)
		x = ProjectedX()
		y = ProjectedY()
		DrawText systemname, x + 100, y - 100 - TextHeight("X")
		DrawLine(x, y, x + 100, y - 100)
		
	EndIf

End Function


' ------------------------------------------------------------------------------------------------
' Initializes the MAXGUI GUI components
' ------------------------------------------------------------------------------------------------
Function InitGUI()

	' get all queries from database
	db = LoadDatabase("SQLITE", "db/sqlqueries.db")
	q = db.executeQuery("SELECT * from queries")
	
	textfield = CreateTextField(200, 0, 800, 24, MAXGUI_CANVAS)
	SetTextAreaFont(textfield, gadgetfont)

	buttonquery = CreateButton("Send Query", 1000, 0, 100, 24, MAXGUI_CANVAS, BUTTON_OK)
	buttonloadcmdr = CreateButton("Load CMDR Database", 1100, 0, 150, 24, MAXGUI_CANVAS, BUTTON_OK)
	buttonloadtex = CreateButton("Load Galaxy Texture", 1250, 0, 150, 24, MAXGUI_CANVAS, BUTTON_OK)

	gadgetfont = LoadGuiFont(GUIFONT_MONOSPACED, 12, True)
	pulldown = CreateComboBox(0, 0, 200, 24, MAXGUI_CANVAS, 0)
	
	While q.nextRow()
	
		Local e:TESAQuery = New TESAQuery
		Local r:TQueryRecord = q.rowRecord()
		
		e.id = r.getIntByName("id")
		e.title = r.getStringByName("title")
		e.desc = r.getStringByName("description")
		e.sql = r.getStringByName("sql")
		
		Local sel:Int = False
		If e.id = selectedquery Then
		
			sel = True
			SetGadgetText(textfield, e.sql)
			RedrawGadget(textfield)
			desc = e.desc
			
		EndIf
		
		MapInsert(querymap, String(e.id), e)
		AddGadgetItem pulldown, e.title, sel
		
	Wend
	

End Function



' ------------------------------------------------------------------------------------------------
' Initializes the Scenery
' ------------------------------------------------------------------------------------------------
Function InitScene()

	SetImageFont(MEDIA.GetFont("EURO"))
	'CAMERA.Center()
		
End Function



' ------------------------------------------------------------------------------------------------
' Initialize
' ------------------------------------------------------------------------------------------------
Function InitTour(cmdr:String)

	Local cnt:Int

	' clear old lists
	ClearList tourlist
	ClearList list

	' delete old Mesh
	If TMesh(xmesh) Then FreeEntity xmesh
	xmesh = CreateMesh(pivot)
	xsurf = CreateSurface(xmesh)
	EntityFX xmesh, 1 + 2
	PositionEntity xmesh, 0, 0, 0
	EntityOrder xmesh, -1000
	EntityTexture xmesh, MEDIA.GetTexture("DOT")
	TextureBlend MEDIA.GetTexture("DOT"), 5
	
	' close old DB connecton
	If TDBConnection(db) Then db.close()
	
	' collect Garbage
	qx = Null
	GCCollect()

	' get all systems from database
	db = LoadDatabase("SQLITE", cmdr)
	q = db.executeQuery("SELECT * from system ORDER by lastvisit ASC")
	
	While q.nextRow()
	
		Local p:TPath = New TPath
		
		Local r:TQueryRecord = q.rowRecord()
		
		p.id = cnt
				
		p.x = r.getFloatByName("x")
		p.y = r.getFloatByName("y")
		p.z = r.getFloatByName("z")
		p.starname = r.getStringByName("name")
		
		qx = New TQuad
		qx.id = cnt
		qx.updatetime = MilliSecs()
		
		qx.x = p.x / elitescaler
		qx.y = p.y / elitescaler
		qx.z = p.z / elitescaler
		
		qx.distance = 1000
		
		qx.mesh = xmesh
		qx.surf = xsurf
		
		qx.RGB = [255, 0, 0]
		qx.cat = p.starname
		qx.Add(0, 0, 0, True)
		
		p.q = qx
		
		ListAddLast(tourlist, qx)
		ListAddLast(list, p)
				
		cnt:+1
	
	Wend
	
	' at init or no DB loaded just create a SOL tour
	If tourlist.count() = 0 Or list.count() = 0 Then
	
		Local p:TPath = New TPath
		p.id = 0
		p.x = 0
		p.y = 0
		p.z = 0
		p.starname = "Sol"
		
		qx = New TQuad
		qx.id = cnt
		qx.updatetime = MilliSecs()
		
		qx.x = 0
		qx.y = 0
		qx.z = 0
		
		qx.distance = 1000
		
		qx.mesh = xmesh
		qx.surf = xsurf
		
		qx.RGB = [255, 0, 0]
		qx.cat = p.starname
		qx.Add(0, 0, 0, True)
		
		p.q = qx
		
		ListAddLast(tourlist, qx)
		ListAddLast(list, p)
	
	EndIf
	
	mx = cnt
	cn = mx

End Function



' ------------------------------------------------------------------------------------------------
' Reset the current Tour and highlight a new SQL query
' ------------------------------------------------------------------------------------------------
Function ResetTour(qu:String)

	' clear lists
	ClearList systemlist
	ClearList vislist
	
	' get all systems who match the query
	q = db.executeQuery(qu)
	While q.nextRow()
		
		Local r:TQueryRecord = q.rowRecord()
		ListAddLast(systemlist, r.getStringByName("systemname"))
	
	Wend
	
	' compare queried systems with existing systems drawn and mark these green
	For Local p:TPath = EachIn list
	
		' mark star systems red by default
		p.q.UpdateVertexColors(255, 0, 0, 1)
	
		' mark matches green and add to vislist
		If ListContains(systemlist, p.starname) Then
		
			p.q.cat = p.starname
			p.q.UpdateVertexColors(0, 255, 0, 1)
			ListAddLast(vislist, p.q)
			
		EndIf
	
	Next

End Function



' ------------------------------------------------------------------------------------------------
' Create or recreates the Galaxy object
' ------------------------------------------------------------------------------------------------
Function CreateGalaxy(tex:String = Null)

	If TEntity(galaxy) Then FreeEntity galaxy
	If TEntity(raster1) Then FreeEntity raster1
	If TEntity(raster2) Then FreeEntity raster2
	If TEntity(raster3) Then FreeEntity raster3
	
	galaxy = CreateCube(pivot)
	EntityFX galaxy, 1 + 32
	ScaleEntity galaxy, -512, 0, 512
	PositionEntity galaxy, 0, 0, elitediff
	
	raster1 = CreateCube(pivot)
	EntityFX raster1, 1
	ScaleEntity raster1, -512, 0, 512
	PositionEntity raster1, 0, 0, elitediff
	EntityBlend raster1, 3
	EntityColor raster1, 128, 128, 128
	
	raster2 = CreateCube(pivot)
	EntityFX raster2, 1
	ScaleEntity raster2, -512, 0, 512
	PositionEntity raster2, 0, 0, elitediff
	EntityBlend raster2, 3
	EntityColor raster2, 192, 192, 192

	raster3 = CreateCube(pivot)
	EntityFX raster3, 1
	ScaleEntity raster3, -512, 0, 512
	PositionEntity raster3, 0, 0, elitediff
	EntityBlend raster3, 3
	EntityColor raster3, 255, 255, 255
	
	EntityAlpha raster1, 1
	EntityAlpha raster2, 0.5
	EntityAlpha raster3, 0.25
	
	If tex Then galaxytex = LoadTexture(tex, 1 + 8 + 16 + 32)
	
	If TTexture(galaxytex) Then EntityTexture galaxy, galaxytex
	RotateEntity galaxy, 0, 180, 0

	If TTexture(MEDIA.GetTexture("RASTER1")) Then
	
		EntityTexture raster1, MEDIA.GetTexture("RASTER1")
		TextureBlend MEDIA.GetTexture("RASTER1"), 5
		ScaleTexture MEDIA.GetTexture("RASTER1"), 1.0 / uvs, 1.0 / uvs
		
	EndIf
	
	If TTexture(MEDIA.GetTexture("RASTER2")) Then
	
		EntityTexture raster2, MEDIA.GetTexture("RASTER2")
		TextureBlend MEDIA.GetTexture("RASTER2"), 5
		ScaleTexture MEDIA.GetTexture("RASTER2"), 1.0 / (uvs * 0.1), 1.0 / (uvs * 0.1)
		
	EndIf
	
	If TTexture(MEDIA.GetTexture("RASTER3")) Then
	
		EntityTexture raster3, MEDIA.GetTexture("RASTER3")
		TextureBlend MEDIA.GetTexture("RASTER3"), 5
		ScaleTexture MEDIA.GetTexture("RASTER3"), 1.0 / (uvs * 0.01), 1.0 / (uvs * 0.01)
		
	EndIf
	
	PositionTexture MEDIA.GetTexture("RASTER1"), uvx, uvy
	PositionTexture MEDIA.GetTexture("RASTER2"), uvx * 0.1, uvy * 0.1
	PositionTexture MEDIA.GetTexture("RASTER3"), uvx * 0.01, uvy * 0.01
	
	EntityOrder raster1, -1
	EntityOrder raster2, -2
	EntityOrder raster3, -3

End Function



' ------------------------------------------------------------------------------------------------
' Create the Galaxy, Nebulae, Dust, Stars, Raster
' ------------------------------------------------------------------------------------------------
Function InitGalaxy()

	Local qs:TQuad
	Local qn:TQuad
	Local qd:TQuad
	Local ss:Int = Ceil(512.0 / ((2 * stp) + 1))
	Local cnt:Int = 0
	
	pivot = CreatePivot()
	PositionEntity pivot, 0, 0, 0
	
	' Create Player
	player = CreateSphere(4, PIVOT)
	EntityFX player, 1
	EntityColor player, 255, 0, 255
	EntityAlpha player, 0.5
	ScaleEntity player, 0.1, 0.1, 0.1
	EntityOrder player, -1
	PointEntity CAMERA.cam, player	
	
	cmesh = CreateMesh(pivot)
	csurf = CreateSurface(cmesh)
	EntityFX cmesh, 1 + 2
	PositionEntity cmesh, 0, 0, 0
	EntityOrder cmesh, -1000
	
	' Stars
	qmesh = CreateMesh(pivot)
	qsurf = CreateSurface(qmesh)
	EntityFX qmesh, 1 + 2
	EntityBlend qmesh, 3
	PositionEntity qmesh, 0, 0, elitediff
	
	' Nebulae
	nmesh = CreateMesh(pivot)
	nsurf = CreateSurface(nmesh)
	EntityFX nmesh, 1 + 2 + 32
	EntityBlend nmesh, 3
	EntityOrder nmesh, -1
	PositionEntity nmesh, 0, 0, elitediff
	
	' Dust
	dmesh = CreateMesh(pivot)
	dsurf = CreateSurface(dmesh)
	EntityFX dmesh, 1 + 2 + 32
	EntityBlend dmesh, 1
	PositionEntity dmesh, 0, 0, elitediff
	
	' Textures
	EntityTexture qmesh, MEDIA.GetTexture("STAR")
	TextureBlend MEDIA.GetTexture("STAR"), 5
	ScaleTexture MEDIA.GetTexture("NEBULA"), 4, 4
	EntityTexture nmesh, MEDIA.GetTexture("NEBULA")
	TextureBlend MEDIA.GetTexture("NEBULA"), 3
	ScaleTexture MEDIA.GetTexture("DUST"), 4, 4
	EntityTexture dmesh, MEDIA.GetTexture("DUST")
	TextureBlend MEDIA.GetTexture("DUST"), 5
	
	bubble = CreateSphere(32)
	EntityFX bubble, 1 + 16 + 32
	EntityOrder bubble, -1
	EntityColor bubble, 0, 128, 255
	ScaleEntity bubble, 5, 5, 5
	EntityAlpha bubble, 0.125
	EntityBlend bubble, 3
	FlipMesh bubble
	
	center = CreateSphere(32)
	EntityFX center, 1 + 16 + 32
	EntityOrder center, -1
	EntityColor center, 0, 0, 0
	ScaleEntity center, 1, 1, 1
	EntityAlpha center, 1
	EntityBlend center, 1
	FlipMesh center
	PositionEntity center, 25.21875 / elitescaler, -20.90625 / elitescaler, 25899.96875 / elitescaler
	
	CreateGalaxy(initialgalaxytex)
		
	' stars
	For Local i:Int = 1 To STARS
		
		qs = New TQuad
		qs.mesh = qmesh
		qs.surf = qsurf
		
		qs.id = cnt
		qs.updatetime = MilliSecs()
		
		qs.x = Rnd(-ss, ss)
		qs.y = Rnd(Rnd(-ss * 0.75, -ss * 0.25), Rnd(ss * 0.25, ss * 0.75))
		qs.z = Rnd(-ss, ss)
		
		Local c:Float = Noise3D((qs.x + ss) / (2 * ss), (qs.y + ss * 0.5) / (2 * ss * 0.5), (qs.z + ss) / (2 * ss), 42, 3, 0.79, 0.125)
		c = Normalize(c, -0.5, 0.5, 0.1, 1)
		
		Local s:Float = Rnd(0.1, 1.0)
				
		qs.scalex = (1.0 / 32 + (1.0 / 16 * c)) * s
		qs.scaley = (1.0 / 32 + (1.0 / 16 * c)) * s
		
		Local col:Int = 0
		Local row:Int = 0
		
		' star distribution
		If Rnd(1) > 0.70 Then
		
			row = 1
				
			If Rnd(1) > 0.7 Then
			
				row = Rand(2, 3)
				If Rnd(1) > 0.7 Then row = Rand(4, 7)
				
			EndIf
			
		EndIf
		
		If Rnd(1) > 0.99 Then col = 1
			
		' scale them randomly
		If row > 0 Then
	
			qs.scalex:*Sqr(row * s)
			qs.scaley:*Sqr(row * s)
			
		EndIf
		
		col = 1
		
		Select row
		
			Case 0 qs.RGB = [128, 16, 8]	' M
			Case 1 qs.RGB = [64, 32, 0]		' K
			Case 2 qs.RGB = [96, 64, 32]	' G
			Case 3 qs.RGB = [128, 128, 64]	' F
			Case 4 qs.RGB = [160, 160, 160] ' A
			Case 5 qs.RGB = [32, 96, 192]	' B
			Case 6 qs.RGB = [48, 48, 255]	' O
			Case 7 qs.RGB = [192, 128, 255]	' W
	
		End Select
		
		qs.RGB = [255, 255, 255]
		
		' add to list
		qs.Add(col, 0, False)
		ListAddLast(starlist, qs)
		
		cnt:+1

	Next
	
	cnt = 0
	
	' nebulae
	For Local i:Int = 1 To nebulae
	
		qn = New TQuad
		qn.mesh = nmesh
		qn.surf = nsurf
		
		qn.id = cnt
		qn.updatetime = MilliSecs()
	
		Local arms:Int = 2
		Local Range:Float = 512.0
		Local spread:Float = 80.0
		Local rot:Float = 2.0
	
		Local multi:Float
		Local turb:Float
		Local dist:Float
		Local bulge:Float
		
		Local angle:Int = Int(Floor(i * 1.0 / (nebulae / arms))) * (360.0 / arms) - 45
		
		' distance
		multi = Rnd(0.5, 1)
		dist = Rnd(0, Range) * Rnd(1, Rnd(Rnd(multi)))
		
		' distance And turbulence	
		turb = Rnd(0, Rnd(spread))
		If Rnd(1) > 0.5 Then turb = -turb
			
		' star position x/z
		qn.x = dist * Cos(angle + (dist * rot)) + Rnd(Rnd(Rnd(-spread)), Rnd(Rnd(spread)))
		qn.z = dist * Sin(angle + (dist * rot)) + Rnd(Rnd(Rnd(-spread)), Rnd(Rnd(spread)))
	
		' star position y
		bulge = Normalize(Distance2D(qn.x, qn.z, 0, 0), 0, Range / 2.0, 0, 180) / 2.0
		If bulge > 90 Then bulge = 90
		qn.y = (Cos(bulge) * Rnd(Rnd(-spread), Rnd(spread)) / 5.0) + (turb / 10.0)
	
		' center / arm relation
		If dist < Range / 2.0 Then
		
			Local col1:Int[] = [255, 0, 0]
			Local col2:Int[] = [255, 255, 0]
			
			If Rnd(1) > 0.5 Then qn.RGB1 = col1 Else qn.RGB1 = col2
			If Rnd(1) > 0.5 Then qn.RGB2 = col1 Else qn.RGB2 = col2
			If Rnd(1) > 0.5 Then qn.RGB3 = col1 Else qn.RGB3 = col2
			If Rnd(1) > 0.5 Then qn.RGB4 = col1 Else qn.RGB4 = col2
			
			Local am:Float = 1.0 / Sqr(dist)
			qn.Alpha1 = Rnd(0, am)
			qn.Alpha2 = Rnd(0, am)
			qn.Alpha3 = Rnd(0, am)
			qn.Alpha4 = Rnd(0, am)
		
		Else
		
			Local col1:Int[] = [255, 0, 0]
			Local col2:Int[] = [0, 0, 255]
			
			If Rnd(1) > 0.5 Then qn.RGB1 = col1 Else qn.RGB1 = col2
			If Rnd(1) > 0.5 Then qn.RGB2 = col1 Else qn.RGB2 = col2
			If Rnd(1) > 0.5 Then qn.RGB3 = col1 Else qn.RGB3 = col2
			If Rnd(1) > 0.5 Then qn.RGB4 = col1 Else qn.RGB4 = col2
		
			Local am:Float = 2.0 / Sqr(dist)
			qn.Alpha1 = Rnd(0, am)
			qn.Alpha2 = Rnd(0, am)
			qn.Alpha3 = Rnd(0, am)
			qn.Alpha4 = Rnd(0, am)
	
		EndIf
			
		qn.scalex = (1.0 - (1.0 / dist)) * (Range - dist) / 2.0
		If qn.scalex > 20 Then qn.scalex = 20
		If qn.scalex < 5 Then qn.scalex = 5
		qn.scaley = qn.scalex * Rnd(0.75, 1.25)
	    qn.rotation = Rnd(360)
	
		' add to list
		qn.Add(Rand(0, 3), Rand(0, 3), True)
		ListAddLast(nebulalist, qn)
		
		cnt:+1
	
	Next
	
	If Not SHOWNEBULA Then HideEntity qn.mesh
	
	cnt = 0
	
	' dust
	For Local i:Int = 1 To dust
	
		qd = New TQuad
		qd.mesh = dmesh
		qd.surf = dsurf
		
		qd.id = cnt
		qd.updatetime = MilliSecs()
	
		Local arms:Int = 2
		Local Range:Float = 512.0
		Local spread:Float = 80.0
		Local rot:Float = 2.0
	
		Local multi:Float
		Local turb:Float
		Local dist:Float
		Local bulge:Float
		
		Local angle:Int = Int(Floor(i * 1.0 / (dust / arms))) * (360.0 / arms) - 45
		
		' distance
		multi = Rnd(0.5, 1)
		dist = Rnd(Range * 0.5, Range) * Rnd(1, Rnd(Rnd(multi)))
		
		' distance And turbulence	
		turb = Rnd(0, Rnd(spread))
		If Rnd(1) > 0.5 Then turb = -turb
			
		' star position x/z
		qd.x = dist * Cos(angle + (dist * rot)) + Rnd(Rnd(Rnd(-spread)), Rnd(Rnd(spread)))
		qd.z = dist * Sin(angle + (dist * rot)) + Rnd(Rnd(Rnd(-spread)), Rnd(Rnd(spread)))
	
		' star position y
		bulge = Normalize(Distance2D(qd.x, qd.z, 0, 0), 0, Range / 2.0, 0, 180) / 2.0
		If bulge > 90 Then bulge = 90
		qd.y = (Cos(bulge) * Rnd(Rnd(-spread), Rnd(spread)) / 10.0) + (turb / 10.0)
		
		Local col1:Int[] = [32, 32, 32]
		Local col2:Int[] = [96, 96, 96]
		
		If Rnd(1) > 0.5 Then qd.RGB1 = col1 Else qd.RGB1 = col2
		If Rnd(1) > 0.5 Then qd.RGB2 = col1 Else qd.RGB2 = col2
		If Rnd(1) > 0.5 Then qd.RGB3 = col1 Else qd.RGB3 = col2
		If Rnd(1) > 0.5 Then qd.RGB4 = col1 Else qd.RGB4 = col2
		
		Local am:Float = 2.0 / Sqr(dist)
		qd.Alpha1 = Rnd(0, am)
		qd.Alpha2 = Rnd(0, am)
		qd.Alpha3 = Rnd(0, am)
		qd.Alpha4 = Rnd(0, am)
	
		qd.scalex = (1.0 - (1.0 / dist)) * (Range - dist) / 2.0
		If qd.scalex > 20 Then qd.scalex = 20
		If qd.scalex < 5 Then qd.scalex = 5
		qd.scaley = qd.scalex * Rnd(0.25, 1.75)
	    qd.rotation = 0
	
		' add to list
		qd.Add(Rand(0, 3), Rand(0, 3), True)
		
		ListAddLast(dustlist, qd)
		
		cnt:+1
	
	Next
	
	If Not SHOWDUST Then HideEntity qd.mesh
	
	cnt = 0
	
	For Local x:Int = -stp To stp Step 1
		
		For Local z:Int = -stp To stp Step 1
		
			qmeshes[x + stp, 0, z + stp] = CopyEntity(qmesh, PIVOT)
			
			PositionEntity qmeshes[x + stp, 0, z + stp], x * ss * 2, 0, (z * ss * 2) + elitediff
			
		Next
	
	Next
	
	PositionMesh qmesh, Rnd(-10, 10), Rnd(-10, 10), Rnd(-10, 10)
	
	If Not SHOWSTARS Then HideEntity qmesh

End Function